package ga.view;

import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.view.appstate.SceneState;
import ga.view.appstate.menu.IMenuListenerParent;
import ga.view.appstate.menu.MenuListener;
import ga.view.appstate.menu.MenuState;
import ga.view.config.GASettings;
import ga.view.config.ViewSettings;
import ga.view.interfaces.IPreEvaluationState;

import java.io.File;
import java.io.FileInputStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme3.app.SimpleApplication;
import com.jme3.app.state.AppState;
import com.jme3.input.KeyInput;
import com.jme3.input.controls.ActionListener;
import com.jme3.input.controls.KeyTrigger;
import com.jme3.system.AppSettings;

/**
 * Main class of furny. This provides a menu to select GA configurations and
 * handles all the work.
 * <p />
 * It should be renamed for future use.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class MTIEvaluator extends SimpleApplication implements MenuListener {
  private static final Logger LOGGER = Logger.getLogger(MTIEvaluator.class
      .getSimpleName());
  private boolean showStatistics;

  private MenuState menu;

  private final String viewSettingsFileName;
  private final File viewSettingsFile;
  private ViewSettings viewSettings;

  private GASettings actSettings;
  private AppState actState;

  /**
   * Creates a new evaluation menu and loads the settings from the given file.
   * 
   * @param viewSettingsFileName
   *          The name of the xml file to load.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public MTIEvaluator(final String viewSettingsFileName) {
    this.viewSettingsFile = null;
    this.viewSettingsFileName = viewSettingsFileName;
  }

  /**
   * Creates a new evaluation menu and loads the settings from the given file.
   * 
   * @param viewSettingsFile
   *          The xml file to load.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public MTIEvaluator(final File viewSettingsFile) {
    this.viewSettingsFile = viewSettingsFile;
    this.viewSettingsFileName = null;
  }

  @Override
  public void simpleInitApp() {
    if (viewSettingsFile != null) {
      viewSettings = ViewSettings.buildFromXml(viewSettingsFile);
    } else {
      viewSettings = ViewSettings.buildFromXml(viewSettingsFileName);
    }

    inputManager.setCursorVisible(!settings.getBoolean("ShowCursor"));
    org.lwjgl.input.Mouse.setGrabbed(!settings.getBoolean("ShowCursor"));

    setShowStatistics(settings.getBoolean("ShowStatistics"));

    inputManager.addMapping("showStats", new KeyTrigger(KeyInput.KEY_S));
    inputManager.addListener(new StatisticsListener(), "showStats");

    inputManager.deleteMapping("SIMPLEAPP_Exit");

    inputManager.addMapping("exit", new KeyTrigger(KeyInput.KEY_ESCAPE));
    inputManager.addListener(new EscapeListener(), "exit");

    flyCam.setEnabled(false);

    menu = new MenuState(this, viewSettings);

    setAndActivate(menu);
  }

  @Override
  public void modelSelected(final String name) {
    menu.setEnabled(false);
    actSettings = viewSettings.get(name);

    // resets the settings if they has run before
    if (actSettings.hasRun()) {
      actSettings.reset();
    }

    actSettings.setHasRun(true);

    AppState newState = null;

    newState = actSettings.get(GASettings.PRE_EVALUATION_STATE);

    if (newState == null) {
      newState = actSettings.get("evaluator");
    }

    setAndActivate(newState);
  }

  @Override
  public void nextState() {
    LOGGER.info(actState.getClass().getSimpleName() + " invoking next state");
    actState.setEnabled(false);

    AppState newState = null;

    if (actState instanceof IPreEvaluationState) {
      newState = actSettings.get(GASettings.EVALUATOR);
    } else if (actState instanceof IInteractiveFitnessEvaluator<?>) {
      newState = actSettings.get(GASettings.POST_EVALUATION_STATE);
    }

    if (newState == null) {
      newState = menu;
    }

    setAndActivate(newState);
  }

  /**
   * Attaches a new app state and enables it.
   * 
   * @param newState
   *          The new app state.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void setAndActivate(final AppState newState) {
    actState = newState;

    if (actState instanceof IMenuListenerParent) {
      ((IMenuListenerParent) actState).setMenuListener(this);
    }

    stateManager.attach(actState);
    actState.setEnabled(true);
  }

  @Override
  public void backToMenu() {
    LOGGER.info(actState.getClass().getSimpleName() + " exiting to main menu");
    if (actState != menu) {
      actState.setEnabled(false);

      setAndActivate(menu);
    } else {
      LOGGER.info("Exiting application");

      for (final GASettings settings : viewSettings) {
        final SceneState state = settings.get("evaluator");
        if (state != null) {
          state.finalCleanUp();
        }
      }
      stop();

      // System.exit(0);
    }
  }

  @Override
  public void simpleUpdate(final float tpf) {

  }

  /**
   * Toggles the statistics display.
   * 
   * @param showStatistics
   *          Show statistics if <code>true</code>.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setShowStatistics(final boolean showStatistics) {
    this.showStatistics = showStatistics;
    LOGGER.log(Level.INFO, "show statistics - " + showStatistics);
    if (showStatistics) {
      guiViewPort.attachScene(guiNode);
    } else {
      guiViewPort.detachScene(guiNode);
    }
  }

  /**
   * Keyboard listener to toggle statistics.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class StatisticsListener implements ActionListener {

    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        setShowStatistics(!showStatistics);
      }
    }
  }

  /**
   * Keyboard listener to catch the ESCAPE event.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class EscapeListener implements ActionListener {
    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        backToMenu();
      }
    }
  }

  /**
   * Main method to test the evaluator.
   * 
   * @param args
   *          No arguments required.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final MTIEvaluator app = new MTIEvaluator(
        "ga/view/example/SimpleObjectGA.xml");

    final AppSettings settings = new AppSettings(true);
    settings.setResolution(1280, 1024);

    try {
      final FileInputStream fis = new FileInputStream("settings.properties");
      settings.load(fis);
      fis.close();
      app.setShowSettings(false);
    } catch (final Exception e) {
      settings.setResolution(1024, 768);
      settings.put("ShowCursor", Boolean.TRUE);
    }

    app.setSettings(settings);

    app.start();
  }
}
